#ifndef FIXFLD_H
#define FIXFLD_H
#include "stdlib.h"
#include "iostream.h"
#include "fixlen.h"
#include "string.h"
class FixedFieldBuffer: public FixedLengthBuffer
// Abstract class designed to support fixed length records
// Use of this class requires that all fields be defined before
//    reading and writing can take place
{
   public:
	FixedFieldBuffer (int maxFields, int RecordSize = 1000);
	FixedFieldBuffer (int maxFields, int * fieldSize);
	// initialize all fields at once
	FixedFieldBuffer (const FixedFieldBuffer &); //copy constructor
	FixedFieldBuffer & operator = (const FixedFieldBuffer &);
	void Clear (); // clear values from buffer
	int AddField (int fieldSize); // define the next field
	int ReadHeader (istream &); // write a buffer to the stream
	int WriteHeader (ostream &) const; // write a buffer to the stream
	int Pack (const void * field, int size = -1); // set the value of the next field of the buffer;
	int Unpack (void * field, int maxBytes = -1); // extract the value of the next field of the buffer
	void Print (ostream &) const;
	int NumberOfFields () const; // return number of defined fields
	int Init (int maxFields);
	int Init (int numFields, int * fieldSize);
 protected:
	int * FieldSize; // array to hold field sizes
	int MaxFields; // maximum number of fields
	int NumFields; // actual number of defined fields
	int NextField; // index of next field to be packed/unpacked
};

inline FixedFieldBuffer :: FixedFieldBuffer (const FixedFieldBuffer & buffer)
	: FixedLengthBuffer (buffer)
{
	Init (buffer . NumFields, buffer . FieldSize);
}

#endif

// public members
FixedFieldBuffer :: FixedFieldBuffer (int maxFields, int maxBytes)
 // construct with a maximum of maxFields
: FixedLengthBuffer(maxBytes)
{
  Init (maxFields);
}

// private function to calculate the record size from the field sizes
static int SumFieldSizes (int numFields, int * fieldSize)
{
	int sum = 0;
	for (int i = 0; i < numFields; i++)
		sum += fieldSize[i];
	return sum;
}

FixedFieldBuffer & FixedFieldBuffer :: operator =
	(const FixedFieldBuffer & buffer)
{
	// disallow copy unless fields are identical
	if (NumFields != buffer . NumFields) return *this;
	for (int i = 0; i < NumFields; i++)
		if (FieldSize[i] != buffer . FieldSize[i]) return *this;
	NextField = buffer . NextField;
	FixedLengthBuffer :: operator = (buffer);
	return *this;
}

FixedFieldBuffer :: FixedFieldBuffer (int numFields, int * fieldSize)
// construct with fields of specific size
: FixedLengthBuffer(SumFieldSizes(numFields, fieldSize))
{
	Init (numFields, fieldSize);
}

int FixedFieldBuffer :: NumberOfFields () const
// return number of fields
{
	return NumFields;
}

void FixedFieldBuffer :: Clear ()
// clear fields from buffer
{
	FixedLengthBuffer::Clear ();
	NextField = 0;
	Buffer[0]=0;
	Packing = TRUE;
}

int FixedFieldBuffer :: AddField (int fieldSize)
{
	Initialized = TRUE;
	if (NumFields == MaxFields) return FALSE;
	if (BufferSize + fieldSize > MaxBytes) return FALSE;
	FieldSize[NumFields] = fieldSize;
	NumFields ++;
	BufferSize += fieldSize;
	return TRUE;
}

int FixedFieldBuffer :: ReadHeader (istream & stream)
// read the header and check for consistency
// see WriteHeader for header record structure
{
	char * str = new char[headerStrSize+1];
	int numFields, *fieldSize;
	int result;
	// read the FixedLengthBufferheader
	result = FixedLengthBuffer::ReadHeader (stream);
	if (result < 0) return -1;
	// read the header string
	stream . read (str, headerStrSize);
	if (!stream.good()) return -1;
	if (strncmp (str, headerStr, headerStrSize) != 0) return -1;
	// read the record description
	stream . read ((char*)&numFields, sizeof(numFields));
	if (!stream) return -1; // failed to read numFields
	fieldSize = new int[numFields];
	for (int i = 0; i < numFields; i ++)
	{
		stream . read ((char*)&fieldSize[i], sizeof(fieldSize[i]));
	}

	if (Initialized) // check header for consistency
	{
		if (numFields != NumFields) return -1;
		for (int j = 0; j < numFields; j ++)
			if (fieldSize[j] != FieldSize[j]) return -1;
		return stream . tellg (); // everything matches
	}
	// else initialize the buffer from the header
	Init (numFields, fieldSize);
	return stream.tellg();
}

int FixedFieldBuffer :: WriteHeader (ostream & stream) const
// write a buffer header to the beginning of the stream
// A header consists of the
//	FixedLengthBufferheader
//	FIXED			5 bytes
//	Variable sized record of length fields
//	that describes the file records
//	Header record size	2 bytes
//	number of fields		4 bytes
//	field sizes			4 bytes per field
{
	int result;
	if (!Initialized) return -1; // cannot write unitialized buffer
	// write the parent (FixedLengthBuffer) header
	result = FixedLengthBuffer::WriteHeader (stream);
	if (!result) return -1;
	// write the header string
	stream . write (headerStr, headerStrSize);
	if (!stream.good()) return -1;
	// write the record description
	stream . write ((char*)&NumFields, sizeof(NumFields));
	for (int i = 0; i < NumFields; i ++)
	{
		stream . write ((char*)&FieldSize[i], sizeof(FieldSize[i]));
	}
	if (!stream) return -1;
	return stream . tellp ();
}

int FixedFieldBuffer :: Pack (const void * field, int size)
// set the value of the next field of the buffer;
//    if size != -1, it must be the same as the packSize
// return number of bytes packed, -1 if error
{
	if (NextField == NumFields || !Packing) // buffer is full or not packing mode
		return -1;
	int start = NextByte; // first byte to be packed
	int packSize = FieldSize[NextField]; // number bytes to be packed
	if (size != -1 && packSize != size) return -1;
	memcpy (&Buffer[start], field, packSize); // move bytes to buffer
	NextByte += packSize;
	NextField ++;
	if (NextField == NumFields) // all fields packed
	{
		Packing = -1;
		NextField = NextByte = 0;
	}
	return packSize;
}

int FixedFieldBuffer :: Unpack (void * field, int maxBytes)
// extract the value of the next field of the buffer
// return the number of bytes extracted, -1 if error
{
	Packing = FALSE;
	if (NextField == NumFields) // buffer is full
		return -1;
	int start = NextByte; // first byte to be unpacked
	int packSize = FieldSize[NextField]; // number bytes to be unpacked
	memcpy (field, &Buffer[start], packSize);
	NextByte += packSize;
	NextField ++;
	if (NextField == NumFields) Clear (); // all fields unpacked
	return packSize;
}

void FixedFieldBuffer :: Print (ostream & stream) const
{
	FixedLengthBuffer::Print (stream);
	stream << endl;
	stream << "\t max fields "<<MaxFields<<" and actual "<<NumFields<<endl;
	for (int i = 0; i < NumFields; i++)
		stream <<"\tfield "<<i<<" size "<<FieldSize[i]<<endl;
	Buffer[BufferSize]=0;
	stream <<"NextByte "<<NextByte<<endl;
	stream <<"Buffer '"<<Buffer<<"'"<<endl;
}

int FixedFieldBuffer :: Init (int maxFields)
 // construct with a maximum of maxFields
{
	Clear ();
	if (maxFields < 0) maxFields = 0;
	MaxFields = maxFields;
	FieldSize = new int[MaxFields];
	BufferSize = 0;
	NumFields = 0;
	return 1;
}

int FixedFieldBuffer :: Init (int numFields, int * fieldSize)
// construct with fields of specific size
{
	// initialize
	Initialized = TRUE;
	Init (numFields);

	// add fields
	for (int j = 0; j < numFields; j++)
		AddField (fieldSize[j]);
	return TRUE;
}


