#ifndef FIXTEXT_H
#define FIXTEXT_H
#include "iostream.h"
#include "string.h"
#include "conio.h"
#ifndef FALSE
#define FALSE (0)
#define TRUE (1)
#endif

class FixedTextBuffer
// a buffer which holds a specific number of fixed sized text fields.
{
 public:
	// construct with a maximum of maxFields
   FixedTextBuffer (int maxFields, int maxChars = 1000);
   FixedTextBuffer (int numFields, int * FieldSize);
   // construct with fields of specific size
   int NumberOfFields () const; // return number of fields
   void Clear (); // clear field values from buffer
   int AddField (int fieldSize);
   int Read (istream &);
   int Write (ostream &);
   int Pack (const char *); // set the value of the next field of the buffer;
   int Unpack (char *); // extract the value of the next field of the buffer
   void Print (ostream &);
   int Init (int numFields, int maxChars = 1000);
   int Init (int numFields, int * fieldSize);
   virtual int DRead (istream &, int recref); // read specified record
   virtual int DWrite (ostream &, int recref) ; // write specified record
   virtual int ReadHeader (istream &); // read the header
   virtual int WriteHeader (ostream &) const; // write the header

 private:
   char * Buffer; // character array to hold field values
   int BufferSize; // sum of the sizes of declared fields
   int * FieldSize; // array to hold field sizes
   int MaxFields; // maximum number of fields
   int MaxChars; // maximum number of characters in the buffer
   int NumFields; // actual number of defined fields
   int NextField; // index of next field to be packed/unpacked
   int NumFieldValues; // number of fields which are packed
   int Packing; // TRUE if in packing phase, FALSE o/w
   int NextCharacter; // packing/unpacking position in buffer
};

#endif

static const char * headerStr = "FixTextBuffer,57";
static const int headerSize = strlen (headerStr);

FixedTextBuffer :: FixedTextBuffer (int maxFields, int maxChars)
 // construct with a maximum of maxFields
{
   Init (maxFields, maxChars);
}

FixedTextBuffer :: FixedTextBuffer (int numFields, int * fieldSize)
// construct with fields of specific size
{
   Init (numFields, fieldSize);
}

int FixedTextBuffer :: NumberOfFields () const
// return number of fields
{
   return NumFields;
}

void FixedTextBuffer :: Clear ()
// clear fields from buffer
{
   NextField = 0;
   NextCharacter = 0;
   Packing = TRUE;
   Buffer[0]=0;
}

int FixedTextBuffer :: AddField (int fieldSize)
{
   if (NumFields == MaxFields)
      return FALSE;
   if (BufferSize + fieldSize > MaxChars)
      return FALSE;
   FieldSize[NumFields] = fieldSize;
   NumFields ++;
   BufferSize += fieldSize;
   return TRUE;
}

int FixedTextBuffer :: Read (istream & stream)
{
   stream . read (Buffer, BufferSize);
   return stream . good ();
}


int FixedTextBuffer :: Write (ostream & stream)
{
   stream . write (Buffer, BufferSize);
   return stream . good ();
}


int FixedTextBuffer :: Pack (const char * str)
// set the value of the next field of the buffer;
{
    // buffer is full or not packing mode
   if (NextField == NumFields || !Packing)
	return FALSE;
  // int len = strlen (str);
   int start = NextCharacter; // first byte to be packed
   int packSize = FieldSize[NextField]; // number bytes to be packed
   strncpy (&Buffer[start], str, packSize);
   NextCharacter += packSize;
   NextField ++;
   // if len < packSize, pad with blanks
   for (int i = start + packSize; i < NextCharacter; i ++)
	Buffer[start] = ' ';
   Buffer [NextCharacter] = 0; // make buffer look like a string
   if (NextField == NumFields) // buffer is full
   {
	Packing = FALSE;
	NextField = NextCharacter = 0;
   }
   return TRUE;
}

int FixedTextBuffer :: Unpack (char * str)
// extract the value of the next field of the buffer
{
   if (NextField == NumFields || Packing) // buffer is full or not unpacking mode
	return FALSE;
   int start = NextCharacter; // first byte to be unpacked
   int packSize = FieldSize[NextField]; // number bytes to be unpacked
   strncpy (str, &Buffer[start], packSize);
   str [packSize] = 0; // terminate string with zero
   NextCharacter += packSize;
   NextField ++;
   if (NextField == NumFields)
       Clear (); // all fields unpacked
   return TRUE;
}

void FixedTextBuffer :: Print (ostream & stream)
{
   stream << "Buffer has max fields "<<MaxFields<<" and actual "<<NumFields<<endl
	<<"max bytes "<<MaxChars<<" and Buffer Size "<<BufferSize<<endl;
   for (int i = 0; i < NumFields; i++)
	stream <<"\tfield "<<i<<" size "<<FieldSize[i]<<endl;
   if (Packing)
       stream <<"\tPacking\n";
   else
      stream <<"\tnot Packing\n";
   stream <<"Contents: '"<< Buffer << "'"<<endl;
}

// Protected members

int FixedTextBuffer :: Init (int maxFields, int maxChars)
 // construct with a maximum of maxFields
{
   if (maxFields < 0)
       maxFields = 0;
   if (maxChars < 0)
       maxChars = 0;
   MaxFields = maxFields;
   MaxChars = maxChars;
   FieldSize = new int[MaxFields];
   Buffer = new char[MaxChars];
   BufferSize = 0;
   NumFields = 0 ;
   NextField = 0;
   Packing = TRUE;
   return 1;
}

int FixedTextBuffer :: Init (int numFields, int * fieldSize)
// construct with fields of specific size
{
   // calculate buffer size
   int bufferSize = 1;
   for (int i = 0; i < numFields; i++)
	bufferSize += fieldSize[i];
	// initialize
   Init (numFields, bufferSize);
	// add fields
   for (int j = 0; j < numFields; j++)
	AddField (fieldSize[j]);
   return 1;
}


int FixedTextBuffer::DRead (istream & stream, int recref)
// read specified record
{
	stream . seekg (recref, ios::beg);
	if (stream . tellg () != recref) return -1;
	return Read (stream);
}

int FixedTextBuffer::DWrite (ostream & stream, int recref)
// write specified record
{
	stream . seekp (recref, ios::beg);
	if (stream . tellp () != recref) return -1;
	return Write (stream);
}


int FixedTextBuffer::ReadHeader (istream & stream)
{
	char str[30];
	stream . seekg (0, ios::beg);
	stream . read (str, headerSize);
	if (! stream . good ()) return -1;
	if (strncmp (str, headerStr, headerSize)==0)
	    return headerSize;
	else return -1;
}

int FixedTextBuffer::WriteHeader (ostream & stream) const
{
	stream . seekp (0, ios::beg);
	stream . write (headerStr, headerSize);
	if (! stream . good ())
	   return -1;
	return headerSize;
}

