using System;             // For String, ArgumentException, Environment
using System.Text;        // For Encoding
using System.IO;          // For IOException 
using System.Net;         // For IPEndPoint, Dns
using System.Net.Sockets; // For TcpClient, NetworkStream, SocketException 
using System.Threading;   // For Thread.Sleep

public class TcpPollEchoClient { 

  static void Main(string[] args) {

    if ((args.Length < 2) || (args.Length > 3)) // Test for correct # of args 
      throw new ArgumentException("Parameters: <Server> <Word> [<Port>]"); 

    String server = args[0]; // Server name or IP address 

    // Convert input String to bytes 
    byte[] byteBuffer = Encoding.ASCII.GetBytes(args[1]);
  
    // Use port argument if supplied, otherwise default to 7 
    int servPort = (args.Length == 3) ? Int32.Parse(args[2]) : 7;

    Socket sock = new Socket(AddressFamily.InterNetwork, SocketType.Stream, 
                             ProtocolType.Tcp);

    IPEndPoint remoteIPEndPoint = new 
                      IPEndPoint(Dns.GetHostByName(server).AddressList[0], 
                                 servPort);

    sock.Connect(remoteIPEndPoint);

    sock.Blocking = false;

    // Receive the same string back from the server 
    int totalBytesSent = 0; // Total bytes sent so far 
    int totalBytesRcvd = 0; // Total bytes received so far 
 
    int bytesRcvd; // Bytes received in last read 

    // Loop until all bytes have been echoed by server
    while (totalBytesRcvd < byteBuffer.Length) { 

      // Send the encoded string to the server 
      if (totalBytesSent < byteBuffer.Length) {  
        try { 
          totalBytesSent += sock.Send(byteBuffer, totalBytesSent, 
                                      byteBuffer.Length - totalBytesSent,  
                                      SocketFlags.None); 

          Console.WriteLine("Sent a total of {0} bytes to server...", totalBytesSent); 

        } catch (SocketException se) { 
          if (se.ErrorCode == 10035) { // WSAEWOULDBLOCK: Resource temporarily unavailable
            Console.WriteLine("Temporarily unable to send, will try again later..."); 
          } else { 
            Console.WriteLine(se.ErrorCode + ": " + se.Message); 
            sock.Close();
            Environment.Exit(se.ErrorCode); 
          } 
        }   
      }
 
      try {
        if (sock.Poll(1000, SelectMode.SelectRead)) { 
          if ((bytesRcvd = sock.Receive(byteBuffer, totalBytesRcvd, 
                byteBuffer.Length - totalBytesRcvd, SocketFlags.None)) == 0) { 
             Console.WriteLine("Connection closed prematurely.");
             break;
          }
          totalBytesRcvd += bytesRcvd; 
        } 
      } catch (SocketException se) {
        Console.WriteLine(se.ErrorCode + ": " + se.Message);
        sock.Close();
        Environment.Exit(se.ErrorCode);
      }
      doThing(); 
    } 
    Console.WriteLine("Received {0} bytes from server: {1}", totalBytesRcvd, 
                      Encoding.ASCII.GetString(byteBuffer, 0, totalBytesRcvd));

    sock.Close(); 
  }

  static void doThing() { 
    Console.Write("."); 
    Thread.Sleep(2000); 
  } 
}
