using System;             // For DateTime, TimeStamp
using System.Net.Sockets; // For Socket
using System.Collections; // For ArrayList
using System.Threading;   // For Thread

public class TimelimitEchoProtocolFactory : IProtocolFactory {

  public TimelimitEchoProtocolFactory() {}

  public IProtocol createProtocol(Socket clntSock, ILogger logger) {
    return new TimelimitEchoProtocol(clntSock, logger);
  }
}

class TimelimitEchoProtocol : IProtocol {
  private static readonly int BUFSIZE   = 32;    // Size (in bytes) of receive buffer
  private static readonly int TIMELIMIT = 10000; // Default time limit (ms)

  private Socket clntSock;
  private ILogger logger;

  public TimelimitEchoProtocol(Socket clntSock, ILogger logger) {
    this.clntSock = clntSock;
    this.logger = logger;
  }

  public void handleclient() {
    ArrayList entry = new ArrayList();
    entry.Add("Client address and port = " + clntSock.RemoteEndPoint);
    entry.Add("Thread = " + Thread.CurrentThread.GetHashCode());

    int totalBytesEchoed = 0;               // Bytes received from client

    try {

      int recvMsgSize;                        // Size of received message
      byte[] echoBuffer = new byte[BUFSIZE];  // Receive Buffer
      DateTime starttime = DateTime.Now;

      TimeSpan elapsed = DateTime.Now - starttime;

      // Set the ReceiveTimeout
      clntSock.SetSocketOption(SocketOptionLevel.Socket,
                               SocketOptionName.ReceiveTimeout, 
                               (int)(TIMELIMIT - elapsed.TotalMilliseconds));

      // Receive until client closes connection or read timeout (both indicated 
      // by SocketException), or TIMELIMIT expired
      while (TIMELIMIT - elapsed.TotalMilliseconds > 0) {
         recvMsgSize = clntSock.Receive(echoBuffer, 0, echoBuffer.Length,
                                        SocketFlags.None);
         clntSock.Send(echoBuffer);
         totalBytesEchoed += recvMsgSize;

         elapsed = DateTime.Now - starttime;
     
         // Set the ReceiveTimeout
         clntSock.SetSocketOption(SocketOptionLevel.Socket,
                                  SocketOptionName.ReceiveTimeout, 
                                  (int)(TIMELIMIT - elapsed.TotalMilliseconds));
      }

      entry.Add("Aborting client, timelimit " + TIMELIMIT + " ms exceeded; echoed " 
                + totalBytesEchoed + " bytes.");

    } catch (SocketException se) {

      if (se.ErrorCode ==  10054) // WSAECONNRESET: Connection reset by peer
        entry.Add("Client finished; echoed " + totalBytesEchoed + " bytes.");

      else if (se.ErrorCode ==  10060) // WSAETIMEDOUT: Connection timed out
        entry.Add("Aborting client, read timed out.");

      else
        entry.Add(se.ErrorCode + ": " +  se.Message);
    } 

    clntSock.Close();

    logger.writeEntry(entry);
  }
}
