using System;             // For String, Int32, Boolean, Console
using System.Text;        // For Encoding
using System.Net;         // For EndPoint, IPEndPoint
using System.Net.Sockets; // For Socket, SocketOptionName, SocketOptionLevel

class UdpEchoClientTimeout {

  private const int TIMEOUT = 3000; // Resend timeout (milliseconds)
  private const int MAXTRIES = 5;   // Maximum retransmissions

  static void Main(string[] args) {

    if ((args.Length < 2) || (args.Length > 3)) { // Test for correct # of args
      throw new ArgumentException("Parameters: <Server> <Word> [<Port>]");
    }

    String server = args[0];     // Server name or IP address

    // Use port argument if supplied, otherwise default to 7
    int servPort = (args.Length == 3) ? Int32.Parse(args[2]) : 7;

    // Create socket that is connected to server on specified port
    Socket sock = new Socket(AddressFamily.InterNetwork, 
                             SocketType.Dgram, ProtocolType.Udp);

    // Set the receive timeout for this socket
    sock.SetSocketOption(SocketOptionLevel.Socket, 
                         SocketOptionName.ReceiveTimeout, TIMEOUT); 

    IPEndPoint remoteIPEndPoint = new 
                     IPEndPoint(Dns.Resolve(server).AddressList[0], servPort);
    EndPoint remoteEndPoint = (EndPoint)remoteIPEndPoint;

    // Convert input String to a packet of bytes 
    byte[] sendPacket = Encoding.ASCII.GetBytes(args[1]);
    byte[] rcvPacket = new byte[sendPacket.Length];

    int tries = 0;  // Packets may be lost, so we have to keep trying
    Boolean receivedResponse = false;

    do {
      sock.SendTo(sendPacket, remoteEndPoint);  // Send the echo string

      Console.WriteLine("Sent {0} bytes to the server...", sendPacket.Length);

      try {
        // Attempt echo reply receive
        sock.ReceiveFrom(rcvPacket, ref remoteEndPoint); 
        receivedResponse = true;
      } catch (SocketException se) {
        tries++;
        if (se.ErrorCode == 10060) // WSAETIMEDOUT: Connection timed out
          Console.WriteLine("Timed out, {0} more tries...", (MAXTRIES - tries)); 
        else // We encountered an error other than a timeout, output error message
          Console.WriteLine(se.ErrorCode + ": " + se.Message);
      }
    } while ((!receivedResponse) && (tries < MAXTRIES));

    if (receivedResponse)
      Console.WriteLine("Received {0} bytes from {1}: {2}", 
                        rcvPacket.Length, remoteEndPoint, 
                        Encoding.ASCII.GetString(rcvPacket, 0, rcvPacket.Length));
    else
      Console.WriteLine("No response -- giving up.");
    
    sock.Close();
  }
}
